package nemosofts.notes.app.activity;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.Intent;
import android.os.Bundle;
import android.os.Handler;
import android.text.Editable;
import android.text.TextWatcher;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.widget.EditText;
import android.widget.FrameLayout;
import android.widget.LinearLayout;
import android.widget.ProgressBar;
import android.widget.TextView;
import android.widget.Toast;

import androidx.activity.EdgeToEdge;
import androidx.activity.OnBackPressedCallback;
import androidx.activity.result.ActivityResultLauncher;
import androidx.activity.result.contract.ActivityResultContracts;
import androidx.core.graphics.Insets;
import androidx.core.view.ViewCompat;
import androidx.core.view.WindowInsetsCompat;
import androidx.nemosofts.AppCompatActivity;
import androidx.nemosofts.utils.DeviceUtils;
import androidx.nemosofts.utils.NetworkUtils;
import androidx.recyclerview.widget.DefaultItemAnimator;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;
import androidx.recyclerview.widget.StaggeredGridLayoutManager;

import com.google.android.gms.tasks.Task;
import com.google.android.play.core.review.ReviewInfo;
import com.google.android.play.core.review.ReviewManager;
import com.google.android.play.core.review.ReviewManagerFactory;
import com.onesignal.Continue;
import com.onesignal.OneSignal;

import java.util.ArrayList;
import java.util.List;

import nemosofts.notes.app.R;
import nemosofts.notes.app.adapters.AdapterCategories;
import nemosofts.notes.app.adapters.NoteAdapter;
import nemosofts.notes.app.callback.Callback;
import nemosofts.notes.app.constant.Constant;
import nemosofts.notes.app.dialog.DialogUtil;
import nemosofts.notes.app.item.Cat;
import nemosofts.notes.app.item.Note;
import nemosofts.notes.app.listeners.ClickListenerPlayList;
import nemosofts.notes.app.utils.AsyncTaskExecutor;
import nemosofts.notes.app.utils.IfSupported;
import nemosofts.notes.app.utils.advertising.AdManagerInterAdmob;
import nemosofts.notes.app.utils.advertising.AdManagerInterApplovin;
import nemosofts.notes.app.utils.advertising.AdManagerInterStartApp;
import nemosofts.notes.app.utils.advertising.AdManagerInterUnity;
import nemosofts.notes.app.utils.advertising.AdManagerInterYandex;
import nemosofts.notes.app.utils.advertising.GDPRChecker;
import nemosofts.notes.app.utils.helper.DBHelper;
import nemosofts.notes.app.utils.helper.Helper;

public class MainActivity extends AppCompatActivity {

    public  static  final int REQUEST_CODE_ADD_NOTE = 1;
    public  static  final int REQUEST_CODE_UPDATE_NOTE = 2;
    private static final int REQUEST_CODE_SHOW_NOTES = 3;
    private RecyclerView notesRecyclerView;
    private List<Note> noteList;
    private NoteAdapter noteAdapter;
    private ActivityResultLauncher<Intent> addNoteLauncher;
    private ActivityResultLauncher<Intent> updateNoteLauncher;

    private Helper helper;
    private DBHelper dbHelper;
    private AdapterCategories adapterCategories;
    private ArrayList<Cat> arrayList;

    private ReviewManager manager;
    private ReviewInfo reviewInfo;

    private ProgressBar pb;
    private String errorMsg;
    private FrameLayout frameLayout;

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        IfSupported.isRTL(this);
        IfSupported.isScreenshot(this);
        EdgeToEdge.enable(this);
        ViewCompat.setOnApplyWindowInsetsListener(findViewById(R.id.bg_page), (v, insets) -> {
            Insets systemBars = insets.getInsets(WindowInsetsCompat.Type.systemBars());
            v.setPadding(systemBars.left, systemBars.top, systemBars.right, systemBars.bottom);
            return insets;
        });

        manager = ReviewManagerFactory.create(this);
        Task<ReviewInfo> request = manager.requestReviewFlow();
        request.addOnCompleteListener(task -> {
            if (task.isSuccessful()) {
                reviewInfo = task.getResult();
            }
        });

        helper = new Helper(this);
        new GDPRChecker(MainActivity.this).check();
        loadAboutData();


        dbHelper = new DBHelper(this);
        arrayList = new ArrayList<>();
        arrayList.addAll(dbHelper.loadPlayList());

        setCategories();
        initActivityResultLaunchers();

        frameLayout = findViewById(R.id.fl_empty);
        pb = findViewById(R.id.pb);

        notesRecyclerView = findViewById(R.id.notesRecyclerView);
        boolean isLandscape = DeviceUtils.isLandscape(MainActivity.this);
        notesRecyclerView.setLayoutManager(new StaggeredGridLayoutManager(isLandscape ? 4 : 2, StaggeredGridLayoutManager.VERTICAL));

        noteList = new ArrayList<>();
        noteAdapter = new NoteAdapter(this, noteList, (note, position) -> {
            Intent intent = new Intent(getApplicationContext(), CreateNoteActivity.class);
            intent.putExtra("isViewOrUpdate", true);
            intent.putExtra("note", note);
            updateNoteLauncher.launch(intent);
        });
        notesRecyclerView.setAdapter(noteAdapter);
        getNotes(REQUEST_CODE_SHOW_NOTES,"");
        try {
            adapterCategories.select(-1);
        } catch (Exception e) {
            Log.e("MainActivity", "Failed to select category", e);
        }

        searchListener();
        findViewById(R.id.button_add).setOnClickListener(view -> openAddListDialog());
        findViewById(R.id.iv_delete).setOnClickListener(view -> startActivity(new Intent(this, DeleteActivity.class)));
        findViewById(R.id.iv_setting).setOnClickListener(view -> startActivity(new Intent(this, SettingActivity.class)));
        findViewById(R.id.imageAddNoteMain).setOnClickListener(view -> addNoteLauncher.launch(
                new Intent(getApplicationContext(), CreateNoteActivity.class)
        ));
        // requestPermission will show the native Android notification permission prompt.
        // NOTE: It's recommended to use a OneSignal In-App Message to prompt instead.
        OneSignal.getNotifications().requestPermission(false, Continue.none());

        setupOnBackPressed();
    }

    private void initActivityResultLaunchers() {
        addNoteLauncher = registerForActivityResult(new ActivityResultContracts.StartActivityForResult(), result -> {
            if (result.getResultCode() == RESULT_OK) {
                getNotes(REQUEST_CODE_ADD_NOTE, "");
            }
        });
        updateNoteLauncher = registerForActivityResult(new ActivityResultContracts.StartActivityForResult(), result -> {
            if (result.getResultCode() == RESULT_OK) {
                getNotes(REQUEST_CODE_UPDATE_NOTE, "");
            }
        });
    }

    private void setCategories() {
        RecyclerView rv = findViewById(R.id.rv_cat);
        LinearLayoutManager layoutManager = new LinearLayoutManager(this, LinearLayoutManager.HORIZONTAL, false);
        rv.setLayoutManager(layoutManager);
        rv.setItemAnimator(new DefaultItemAnimator());
        adapterCategories = new AdapterCategories(this, arrayList, new ClickListenerPlayList() {
            @Override
            public void onClick(int position) {
                adapterCategories.select(Integer.parseInt(arrayList.get(position).id()));
                noteList.clear();
                getNotes(REQUEST_CODE_SHOW_NOTES,arrayList.get(position).id());
            }

            @SuppressLint("NotifyDataSetChanged")
            @Override
            public void onItemZero() {
                arrayList.clear();
                arrayList.addAll(dbHelper.loadPlayList());
                adapterCategories.notifyDataSetChanged();
            }
        });
        rv.setAdapter(adapterCategories);
    }

    private void searchListener() {
        EditText inputSearch = findViewById(R.id.inputSearch);
        inputSearch.addTextChangedListener(new TextWatcher() {
            @Override
            public void beforeTextChanged(CharSequence charSequence, int i, int i1, int i2) {
                // this method is empty
            }

            @Override
            public void onTextChanged(CharSequence charSequence, int i, int i1, int i2) {
                if (noteAdapter == null){
                    return;
                }
                noteAdapter.cancelTimer();
            }

            @Override
            public void afterTextChanged(Editable editable) {
                if (noteAdapter == null || noteList.isEmpty()){
                    return;
                }
                noteAdapter.searchNote(editable.toString());
            }
        });
    }

    private void loadAboutData() {
        if (NetworkUtils.isConnected(this)) {
            helper.initializeAds();
            if (Constant.itemApp.getInterAd() && Constant.itemApp.getAdStatus()) {
                switch (Constant.itemApp.getAdNetwork()) {
                    case Callback.AD_TYPE_ADMOB :
                        AdManagerInterAdmob adManagerInterAdmob = new AdManagerInterAdmob(getApplicationContext());
                        adManagerInterAdmob.createAd();
                        break;
                    case Callback.AD_TYPE_STARTAPP :
                        AdManagerInterStartApp adManagerInterStartApp = new AdManagerInterStartApp(getApplicationContext());
                        adManagerInterStartApp.createAd();
                        break;
                    case Callback.AD_TYPE_APPLOVIN :
                        AdManagerInterApplovin adManagerInterApplovin = new AdManagerInterApplovin(MainActivity.this);
                        adManagerInterApplovin.createAd();
                        break;
                    case Callback.AD_TYPE_YANDEX :
                        AdManagerInterYandex adManagerInterYandex = new AdManagerInterYandex(MainActivity.this);
                        adManagerInterYandex.createAd();
                        break;
                    case Callback.AD_TYPE_UNITY :
                        AdManagerInterUnity adManagerInterUnity = new AdManagerInterUnity();
                        adManagerInterUnity.createAd();
                        break;
                    default:
                        break;
                }
            }
            LinearLayout adView = findViewById(R.id.ll_adView);
            helper.showBannerAd(adView);
        }
    }

    private void setupOnBackPressed() {
        OnBackPressedCallback callback = new OnBackPressedCallback(true) {
            @Override
            public void handleOnBackPressed() {
                if (reviewInfo != null){
                    Task<Void> flow = manager.launchReviewFlow(MainActivity.this, reviewInfo);
                    flow.addOnCompleteListener(task1 -> DialogUtil.exitDialog(MainActivity.this));
                } else {
                    DialogUtil.exitDialog(MainActivity.this);
                }
            }
        };
        getOnBackPressedDispatcher().addCallback(this, callback);
    }

    @Override
    public int setContentViewID() {
        return R.layout.activity_main;
    }

    @SuppressLint("NotifyDataSetChanged")
    private void openAddListDialog() {
        DialogUtil.addCategoriesDialog(this, string -> {
            arrayList.clear();
            arrayList.addAll(dbHelper.addPlayList(string));
            Toast.makeText(MainActivity.this, getString(R.string.add_categories), Toast.LENGTH_SHORT).show();
            adapterCategories.notifyDataSetChanged();
        });
    }

    private void getNotes(final int requestCode, String id){
        new AsyncTaskExecutor<String, String, String>() {

            final List<Note> notes = new ArrayList<>();

            @Override
            protected void onPreExecute() {
                super.onPreExecute();
                frameLayout.setVisibility(View.GONE);
                notesRecyclerView.setVisibility(View.GONE);
                pb.setVisibility(View.VISIBLE);
            }

            @Override
            protected String doInBackground(String strings) {
                try {
                    if (id.isEmpty()){
                        notes.addAll(dbHelper.getAllNotes());
                    } else {
                        notes.addAll(dbHelper.getNotesByCategory(id));
                    }
                    return "1";
                } catch (Exception e) {
                    return "0";
                }
            }

            @SuppressLint("NotifyDataSetChanged")
            @Override
            protected void onPostExecute(String s) {
                if (isFinishing()){
                    return;
                }
                if (notes.isEmpty()){
                    errorMsg = "No Notes";
                    setEmpty(notes);
                    return;
                }
                if (requestCode == REQUEST_CODE_SHOW_NOTES){
                    noteList.addAll(notes);
                    noteAdapter.notifyDataSetChanged();
                } else if (requestCode == REQUEST_CODE_ADD_NOTE){
                    noteList.clear();
                    noteList.addAll(notes);
                    noteAdapter.notifyDataSetChanged();
                    notesRecyclerView.smoothScrollToPosition(0);
                } else if (requestCode == REQUEST_CODE_UPDATE_NOTE){
                    noteList.clear();
                    noteList.addAll(notes);
                    noteAdapter.notifyDataSetChanged();
                }
                errorMsg = "add Data";
                setEmpty(notes);
            }
        }.execute();
    }

    private void setEmpty(List<Note> notes) {
        if (notes== null || !notes.isEmpty()) {
            notesRecyclerView.setVisibility(View.VISIBLE);
            pb.setVisibility(View.INVISIBLE);
            frameLayout.setVisibility(View.GONE);
            return;
        }
        notesRecyclerView.setVisibility(View.GONE);
        frameLayout.setVisibility(View.VISIBLE);
        pb.setVisibility(View.INVISIBLE);

        frameLayout.removeAllViews();
        LayoutInflater inflater = (LayoutInflater) getSystemService(Context.LAYOUT_INFLATER_SERVICE);

        @SuppressLint("InflateParams") View myView = inflater.inflate(R.layout.row_empty, null);

        TextView textView = myView.findViewById(R.id.tv_empty_msg);
        textView.setText(errorMsg);

        myView.findViewById(R.id.ll_empty_try).setOnClickListener(v -> {
            myView.findViewById(R.id.iv_empty).setVisibility(View.GONE);
            myView.findViewById(R.id.pb_empty).setVisibility(View.VISIBLE);
            new Handler().postDelayed(() -> {
                myView.findViewById(R.id.iv_empty).setVisibility(View.VISIBLE);
                myView.findViewById(R.id.pb_empty).setVisibility(View.GONE);
                getNotes(REQUEST_CODE_SHOW_NOTES,"");
            }, 500);
        });
        frameLayout.addView(myView);
    }

    @Override
    public void onResume() {
        if (Callback.isRecreate()) {
            Callback.setRecreate(false);
            recreate();
        }
        super.onResume();
    }
}
